static int socket_descriptor_to_socket(int sock_desc)
{
  int n = MAX_SOCKETS;

  while(n--)
  {
    if(sockets[n] && sockets[n]->socket_descriptor == sock_desc)
      break;
  }
  return n;
}

int socket_exists(int sock_desc);
#pragma aux socket_exists parm [bx] value [ax];
int socket_exists(int sock_desc)
{
  return socket_descriptor_to_socket(sock_desc) + 1;
}

int socket_status(int sock_desc);
#pragma aux socket_status parm [bx] value [ax];
int socket_status(int sock_desc)
{
  int socket = socket_descriptor_to_socket(sock_desc);
  if(socket == -1)
  {
    error_code = ENOTCONN;
    return -1;
  }
  return sockets[socket]->status;
}

unsigned socket_count(void);
#pragma aux socket_count value [ax];
unsigned socket_count(void)
{
  unsigned retval = 0;
  unsigned n = MAX_SOCKETS;
  while(n--)
    if(sockets[n]) retval++;
  return retval;
}

static int socket_descriptor_to_open_port(int sock_desc)
{
  int n = MAX_OPEN_PORTS;

  while(n--)
  {
    if(open_ports[n] && open_ports[n]->socket_descriptor == sock_desc)
      break;
  }
  return n;
}

static int next_socket_descriptor(void)
{
  while(socket_descriptor_to_socket(socket_descriptor_index) != -1
        || socket_descriptor_to_open_port(socket_descriptor_index) != -1
        || socket_descriptor_index < 0)
    socket_descriptor_index++;
    
  return socket_descriptor_index;
}

static int find_free_socket_index(void)
{
  int n = MAX_SOCKETS;
  int desc = next_socket_descriptor();

  while(n--)
    if(!sockets[n])
    {
      sockets[n] = tcpmalloc(sizeof(struct transmission_control_block));
      if(!FP_SEG(sockets[n])) return -1;
      _fmemset(sockets[n], 0, sizeof(struct transmission_control_block));
      sockets[n]->socket_descriptor = desc;
      sockets[n]->ploss = INIT_PLOSS; // slow start...
      break;
    }
  return n;
}

static int is_any_address(void __far *addr)
{
  return !(*(unsigned long __far*)addr);
}

static int is_any_address6(void __far *addr)
{
  unsigned __far *ptr = addr;
  unsigned n = 8;

  while(n--)
    if(*ptr++) return 0;
  return 1;
}

static int find_socket_by_ports_and_remote_addr4(unsigned local_port, unsigned remote_port, struct ipv4_addr address, unsigned udp)
{
  int n = MAX_SOCKETS;

  while(n--)
    if(sockets[n] && sockets[n]->local_port == local_port
       && !sockets[n]->ipv6 && sockets[n]->udp == udp)
       if(is_any_address(&sockets[n]->ip_addr.ipv4_addr) // INADDR_ANY for recvfrom
       || (sockets[n]->remote_port == remote_port
           && (!_fmemcmp(&sockets[n]->ip_addr.ipv4_addr, &address, sizeof(struct ipv4_addr))
               || address.p1 == 127 && sockets[n]->ip_addr.ipv4_addr.p1 == 127)))
      break;

  return n;
}

static int find_socket_by_ports_and_remote_addr6(unsigned local_port, unsigned remote_port, struct ipv6_addr address, unsigned udp)
{
  int n = MAX_SOCKETS;

  while(n--)
    if(sockets[n] && sockets[n]->local_port == local_port
    && sockets[n]->ipv6 && sockets[n]->udp == udp)
      if(is_any_address6(&sockets[n]->ip_addr.ipv6_addr)
      || (sockets[n]->remote_port == remote_port
          && (!_fmemcmp(&sockets[n]->ip_addr.ipv6_addr, &address, sizeof(struct ipv6_addr))
              || is_localhost6(&address) && is_localhost6(&sockets[n]->ip_addr.ipv6_addr))))
    break;

  return n;
}

static unsigned tcp4_verify_incoming_header_sanity(unsigned socket, struct tcp4_header *header)
{
  if(header->dst_port != sockets[socket]->local_port) return 1;

  return 0;
}

static unsigned socket_buffer_free(unsigned socket)
{
  return SOCKET_IN_BUFF_SIZE - (sockets[socket]->in_buff_write_offset - sockets[socket]->in_buff_read_offset);
}

static unsigned get_window_size(unsigned socket)
{
  #define TCP_WINDOW_OVERHEAD 100
  signed raw_buffer_space = raw_buffer_free - TCP_WINDOW_OVERHEAD;
  signed socket_buffer = socket_buffer_free(socket);

  if(raw_buffer_space <= socket_buffer)
    if(raw_buffer_space > 0)
      return raw_buffer_space;
    else
      return 0;
  else
    return socket_buffer;
}

static void tcp4_send_ack(unsigned socket)
{
  struct tcp4_header header;
  
  memset(&header, 0, sizeof(struct tcp4_header));

  header.flags.ack = 1;
  header.window = get_window_size(socket);
  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;
  header.ack_number = sockets[socket]->recv.irs + sockets[socket]->recv.count;
  header.seq_number = sockets[socket]->send.iss + sockets[socket]->send.count;

  if(!header.window) sockets[socket]->recv.zerowindow = 1;
  else sockets[socket]->recv.zerowindow = 0;

  send_tcp(socket, &header, 0, sockets[socket]->remote_addr);      
}


static void tcp4_ack_socket(unsigned socket, struct tcp4_header *synack)
{
  if(synack->ack_number == sockets[socket]->send.iss + 1)
  {
    if(tcp4_verify_incoming_header_sanity(socket, synack))
    {
      return;
    }

    sockets[socket]->recv.irs = synack->seq_number;
    sockets[socket]->send.count = 1;
    sockets[socket]->recv.count = 1;
    tcp4_send_ack(socket);

    sockets[socket]->status = 1;
  }
}

static void tcp4_reack_socket(unsigned socket)
{
  tcp4_send_ack(socket);

  tcp4_packetizer(socket); // also resend payload if any
}

static void tcp4_synack_socket(unsigned socket, struct tcp4_header *syn)
{
  struct tcp4_header header;

  memset(&header, 0, sizeof(struct tcp4_header));

  header.flags.ack = 1;
  header.flags.syn = 1;

  sockets[socket]->recv.irs = syn->seq_number;
  header.ack_number = syn->seq_number + (sockets[socket]->recv.count = 1);
  header.seq_number = sockets[socket]->send.iss;
  sockets[socket]->send.count = 1;
  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;
  header.window = get_window_size(socket);

  send_tcp(socket, &header, 0, sockets[socket]->remote_addr);
}

static void tcp4_resynack_socket(unsigned socket)
{
  struct tcp4_header header;

  memset(&header, 0, sizeof(struct tcp4_header));

  header.flags.ack = 1;
  header.flags.syn = 1;

  header.ack_number = sockets[socket]->recv.irs + 1;
  header.seq_number = sockets[socket]->send.iss;
  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;
  header.window = get_window_size(socket);

  sockets[socket]->time = ttime(0);
  sockets[socket]->send.retries++;

  send_tcp(socket, &header, 0, sockets[socket]->remote_addr);
}

static void udp4_recv_socket(unsigned socket, unsigned char __far *packet, unsigned packet_len)
{
  unsigned __far *write_offset = &sockets[socket]->in_buff_write_offset;
//  unsigned __far *read_offset = &sockets[socket]->in_buff_read_offset;

  if(packet_len <= socket_buffer_free(socket))
  {
    copy2ring(sockets[socket]->in_buff, packet, *write_offset, SOCKET_IN_BUFF_SIZE, packet_len);
    *write_offset += packet_len;
  }
}

static void tcp4_recv_socket(unsigned socket, struct tcp4_header *rheader, unsigned char __far *packet, unsigned packet_len)
{
  unsigned __far *write_offset = &sockets[socket]->in_buff_write_offset;
//  unsigned __far *read_offset = &sockets[socket]->in_buff_read_offset;

  tcp4_verify_send(socket, rheader);

  if(tcp4_verify_incoming_header_sanity(socket, rheader))
  {
    return;
  }


  if(rheader->seq_number != sockets[socket]->recv.irs + sockets[socket]->recv.count)
  {
    // this packet seems to be a retransmission, resend ACK
    if(rheader->seq_number == sockets[socket]->recv.irs + sockets[socket]->recv.count - sockets[socket]->recv.last_len
    && packet_len == sockets[socket]->recv.last_len)
      goto resend_ack;
    // else drop packet
    else return;
  }

  if(packet_len > socket_buffer_free(socket))
  {
// No space left in buffer
    return;
  }

  sockets[socket]->send.window = rheader->window;
  if(!packet_len) return;

  copy2ring(sockets[socket]->in_buff, packet, *write_offset, SOCKET_IN_BUFF_SIZE, packet_len);
  *write_offset += packet_len;

  sockets[socket]->recv.count += packet_len;

  sockets[socket]->recv.last_len = packet_len;

  resend_ack:

  if(!rheader->flags.fin)
    tcp4_send_ack(socket);
  else
    tcp4_remote_close_2(socket, rheader);
}

static int tcp4_push_socket(unsigned socket, struct tcp4_data *data)
{
  struct tcp4_header header;

  memset(&header, 0, sizeof(struct tcp4_header));

  header.flags.push = 1;
  header.flags.ack = 1;
  header.seq_number = sockets[socket]->send.iss + sockets[socket]->send.count;
  header.ack_number = sockets[socket]->recv.irs + sockets[socket]->recv.count;

  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;
  header.window = get_window_size(socket);

  sockets[socket]->status = 2;

  sockets[socket]->time = ttime(0);  

  return send_tcp(socket, &header, data, sockets[socket]->remote_addr);
}

static void delete_socket(unsigned socket, unsigned force)
{
  if(force)
  {
    free_socket:
    tsr_free(sockets[socket]);
    sockets[socket] = 0;    
    socket_delete_buffers(socket);
    return;
  }

  if(sockets[socket]->data_pending && !sockets[socket]->user_closed)
    return;

  if(sockets[socket]->out_buff_write_offset == sockets[socket]->out_buff_read_offset)
    goto free_socket;
  else
    sockets[socket]->send_pending = 1;
}

static void tcp4_close_socket(unsigned socket)
{
  struct tcp4_header header;
  unsigned write_offset = sockets[socket]->out_buff_write_offset;
  unsigned read_offset = sockets[socket]->out_buff_read_offset;

  if(write_offset != read_offset)
  {
    sockets[socket]->send_pending = 1;
    return;
  }

  sockets[socket]->time = ttime(0);

  if(sockets[socket]->status == 3)
  {
    delete_socket(socket, 0);
    return;
  }

  memset(&header, 0, sizeof(struct tcp4_header));

  header.flags.fin = 1;
  header.flags.ack = 1;
  header.seq_number = sockets[socket]->send.iss + sockets[socket]->send.count++;
  header.ack_number = sockets[socket]->recv.irs + sockets[socket]->recv.count;
  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;

  send_tcp(socket, &header, 0, sockets[socket]->remote_addr);

  sockets[socket]->status = 3;
}

static void tcp4_remote_close_2(unsigned socket, struct tcp4_header *rheader)
{
  unsigned write_offset = sockets[socket]->in_buff_write_offset;
  unsigned read_offset = sockets[socket]->in_buff_read_offset;

  if(write_offset != read_offset)
    sockets[socket]->data_pending = 1;

  sockets[socket]->recv.count++;

  tcp4_send_ack(socket);
  tcp4_close_socket(socket);
}

static void tcp4_remote_close(unsigned socket, struct tcp4_header *rheader)
{
  if(rheader->seq_number != sockets[socket]->recv.irs + sockets[socket]->recv.count)
    return;
  tcp4_remote_close_2(socket, rheader);
}

static void tcp4_reset_socket(unsigned socket)
{
  struct tcp4_header header;

  memset(&header, 0, sizeof(struct tcp4_header));

  header.flags.reset = 1;
  header.seq_number = sockets[socket]->send.iss + sockets[socket]->send.count;

  header.dst_port = sockets[socket]->remote_port;
  header.dst_port = sockets[socket]->local_port;

  send_tcp(socket, &header, 0, sockets[socket]->remote_addr);

  delete_socket(socket, 1);  
}

static void send_syn4(int socket)
{
  struct tcp4_header header;

  memset(&header, 0, sizeof(struct tcp4_header));
  header.window = SOCKET_IN_BUFF_SIZE;
  sockets[socket]->send.iss = header.seq_number = random32();
  sockets[socket]->time = ttime(0);
  sockets[socket]->status = 0;
  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;
  header.flags.syn = 1;
  send_tcp(socket, &header, 0, 0);
}

static void resend_syn4(int socket)
{
  struct tcp4_header header;

  memset(&header, 0, sizeof(struct tcp4_header));
  sockets[socket]->time = ttime(0);
  sockets[socket]->send.retries++;
  header.window = SOCKET_IN_BUFF_SIZE;
  header.seq_number = sockets[socket]->send.iss;
  header.dst_port = sockets[socket]->remote_port;
  header.src_port = sockets[socket]->local_port;
  header.flags.syn = 1;
  send_tcp(socket, &header, 0, 0);
}

static int new_tcp_socket(unsigned d, unsigned remote_port, unsigned local_port)
{
  int s = find_free_socket_index();

  if(s != -1)
  {
//    sockets[s]->ip_addr.ipv4_addr = dst;    
    _fmemcpy(&sockets[s]->ip_addr, &dns_cache[d].ip_addr, sizeof(union ip_addr));
    sockets[s]->remote_port = remote_port;
    sockets[s]->local_port = local_port;
    sockets[s]->ipv6 = dns_cache[d].ipv6;
    
    send_syn4(s);
  }
  else error_code = ENOMEM;
  
  return s;
}

static int find_free_open_port_index(void)
{
  int n = MAX_OPEN_PORTS;
  int desc = next_socket_descriptor();

  while(n--)
    if(!open_ports[n])
    {
      open_ports[n] = tcpmalloc(sizeof (struct open_port_control_block));
      if(!FP_SEG(open_ports[n]))
        return -1;
      _fmemset(open_ports[n], 0, sizeof(struct open_port_control_block));
      open_ports[n]->socket_descriptor = desc;
      break;
    }
  
  return n;
}

static int find_open_port_by_port(unsigned port, unsigned udp)
{
  int n = MAX_OPEN_PORTS;

  while(n--)
    if(open_ports[n] && open_ports[n]->local_port == port)
      if(open_ports[n]->udp == udp)
        break;

  return n;
}

int open_port(unsigned local_port, unsigned udp)
{
  int s;
  
  if(find_open_port_by_port(local_port, udp) != -1)
  {
    error_code = EADDRINUSE;
    return -1;
  }

  if((s = find_free_open_port_index()) != -1)
  {
    open_ports[s]->udp = udp;
    open_ports[s]->local_port = local_port;
    return open_ports[s]->socket_descriptor;
  }
  error_code = ENOMEM;
  return -1;
}

static int close_port(unsigned port_desc)
{
  int socket;
  while(open_ports[port_desc]->socket_queue_count)
  {
    socket = pop_socket_queue(port_desc);
    close_socket(socket_descriptor_to_socket(socket));
  }
  tsr_free(open_ports[port_desc]);
  open_ports[port_desc] = 0;
  return 0;
}

static int new_udp_socket(unsigned d, unsigned remote_port, unsigned local_port)
{
  int s = find_free_socket_index();

  if(s != -1)
  {
//    sockets[s]->ip_addr.ipv4_addr = dst;
    _fmemcpy(&sockets[s]->ip_addr, &dns_cache[d].ip_addr, sizeof(union ip_addr));
    sockets[s]->ipv6 = dns_cache[d].ipv6;
    sockets[s]->remote_port = remote_port;
    sockets[s]->local_port = local_port;
    sockets[s]->status = sockets[s]->udp = 1;
  }
  else error_code = ENOMEM;

  return s;
}

// BSD-style functions...

int new_socket(unsigned, unsigned, unsigned);
#pragma aux new_socket parm [bx] [cx] [dx] value [ax];
int new_socket(unsigned protocol, unsigned type, unsigned rawproto)
{
  int s = find_free_socket_index();
  if(s != -1)
  {
    if(type & 0x8000)
    {
      sockets[s]->raw = type;
      sockets[s]->raw_protocol = rawproto;
    }
    else
      sockets[s]->udp = type;
    sockets[s]->ipv6 = protocol;
    return sockets[s]->socket_descriptor;
  }
  else
  {
    error_code = ENOMEM;
    return -1;
  }
}

int bind_socket(unsigned, union sockaddr __far *, unsigned);
#pragma aux bind_socket parm [bx] [es dx] [cx] value [ax];
int bind_socket(unsigned sock_desc, union sockaddr __far *addr_struct, unsigned addrlen)
{
  int socket = socket_descriptor_to_socket(sock_desc);
  union portsock __far *sock;

  if(socket == -1)
  {
    socket = socket_descriptor_to_open_port(sock_desc);
    if(socket == -1)
    {
      error_code = ENOTCONN;
      return -1;
    }
    else
    {
      sock = (union portsock __far*)open_ports[socket];
    }
  }
  else
  {
    sock = (union portsock __far*)sockets[socket];
    sock->sock.status = sock->sock.udp;
  }
  // use sockaddr_in struct for both ipv4 and 6...
  sock->sock.local_port = switch_endianness_s(addr_struct->sockaddr_in.sin_port);

  return 0;
}

static void setpeer(unsigned socket, union sockaddr __far *sockaddr)
{
  sockets[socket]->remote_port = switch_endianness_s(sockaddr->sockaddr_in.sin_port);

  if(!sockets[socket]->ipv6)
    _fmemcpy(&sockets[socket]->ip_addr, sockaddr->sockaddr_in.sin_addr, 4);
  else
    _fmemcpy(&sockets[socket]->ip_addr, sockaddr->sockaddr_in6.sin6_addr, 16); 
}

int setpeername(unsigned, union sockaddr __far *);
#pragma aux setpeername value [ax] parm [bx] [es dx];
int setpeername(unsigned socket_desc, union sockaddr __far *sockaddr)
{
  int socket = socket_descriptor_to_socket(socket_desc);
  if(socket == -1)
  {
    error_code = ENOTCONN;
    return -1;
  }
  setpeer(socket, sockaddr);
  if(is_any_address((void __far*)&sockaddr->sockaddr_in.sin_addr))
    _fmemset(sockets[socket]->remote_addr, 0, 6);
  return 0;
}

int connect_socket(int, union sockaddr __far*);
#pragma aux connect_socket parm [bx] [es dx] value [ax];
int connect_socket(int socket_desc, union sockaddr __far *addr)
{
  int socket = socket_descriptor_to_socket(socket_desc);

  if(socket == -1)
  {
    error_code = ENOTCONN;
    return -1;
  }
  if(sockets[socket]->ipv6 != addr->sockaddr_in.sin_family) // different local and remote address families
  {
    error_code = EAFNOSUPPORT;
    return -1;
  }

  setpeer(socket, addr);
  sockets[socket]->local_port = get_free_port(sockets[socket]->udp,
                                &sockets[socket]->ip_addr.ipv4_addr,
                                sockets[socket]->remote_port, 
                                sockets[socket]->ipv6);
  if(!sockets[socket]->udp && !sockets[socket]->raw)
  {
    send_syn4(socket);
  }
  else
    sockets[socket]->status = 1;

  return 0;
}

int listen_socket(int socket_desc);
#pragma aux listen_socket parm [bx] value [ax];
int listen_socket(int socket_desc)
{
  int socket = socket_descriptor_to_socket(socket_desc);
  int udp;
  int ipv6;
  unsigned port;
  int retval;

  if(socket == -1) return -1;

  udp = sockets[socket]->udp;
  ipv6 = sockets[socket]->ipv6;
  port = sockets[socket]->local_port;

  retval = open_port(port, udp);
  if(retval != -1)
  {
    delete_socket(socket, 1);
//    socket = find_open_port_by_port(port, udp);
    socket = socket_descriptor_to_open_port(retval);
    open_ports[socket]->socket_descriptor = socket_desc;
    open_ports[socket]->ipv6 = ipv6;
    open_ports[socket]->ipv_matters = 1;
    return 0;
  }
  return retval;
}

static int addrlen_check(int ipv6, unsigned addrlen)
{
  if(!ipv6)
  {
    if(addrlen < sizeof(struct sockaddr_in))
      return sizeof(struct sockaddr_in);
  }
  else
  {
    if(addrlen < sizeof(struct sockaddr_in6))
      return sizeof(struct sockaddr_in6);
  }
  return 0;
}

int getpeername(int, union sockaddr __far*, unsigned);
#pragma aux getpeername parm [bx] [es dx] [cx] value [ax];
int getpeername(int sockfd, union sockaddr __far *addr, unsigned addrlen)
{
  int socket = socket_descriptor_to_socket(sockfd);
  struct sockaddr_in __far *sockaddr_in;
  struct sockaddr_in6 __far *sockaddr_in6;
  int rc;

  if(socket == -1)
  {
    error_code = ENOTCONN;
    return -1;
  }
  if(rc = addrlen_check(sockets[socket]->ipv6, addrlen))
    return rc;
  if(!sockets[socket]->ipv6)
  {
    _fmemcpy(addr->sockaddr_in.sin_addr, &sockets[socket]->ip_addr.ipv4_addr, 4);
  }
  else
  {
    _fmemcpy(addr->sockaddr_in6.sin6_addr, &sockets[socket]->ip_addr.ipv6_addr, 16);
  }
  addr->sockaddr_in.sin_family = sockets[socket]->ipv6;
  addr->sockaddr_in.sin_port = switch_endianness_s(sockets[socket]->remote_port);
  return 0;
}

int getsockname(int, union sockaddr __far*, unsigned);
#pragma aux getsockname parm [bx] [es dx] [cx] value [ax];
int getsockname(int sockfd, union sockaddr __far *addr, unsigned addrlen)
{
  int socket = socket_descriptor_to_socket(sockfd);
  int rc;
  union portsock __far *portsock;
  void __far* ptr;
  struct ipv6_addr ipv6_addr;


  if(socket == -1)
  {
    // active socket not found, try to find a passive socket
    socket = socket_descriptor_to_open_port(sockfd);
    if(socket == -1)
    {
      error_code = ENOTCONN;
      return -1;
    }
    portsock = (union portsock __far*)open_ports[socket];
//    if(rc = addrlen_check(open_ports[socket]->ipv6))
//      return rc;
//    addr->sockaddr_in.sin_family = open_ports[socket]->ipv6;
//    addr->sockaddr_in.sin_port = switch_endianness_s(open_ports[socket]->local_port);
//    goto fill_my_ip;
  }
  else portsock = (union portsock __far*)sockets[socket];

  if(rc = addrlen_check(portsock->sock.ipv6, addrlen))
    return rc,

  addr->sockaddr_in.sin_port = switch_endianness_s(portsock->sock.local_port);
  addr->sockaddr_in.sin_family = portsock->sock.ipv6;

//  fill_my_ip:
  if(!addr->sockaddr_in.sin_family)
    _fmemcpy(addr->sockaddr_in.sin_addr, &ipv4_my_addr, 4);
  else
  {
    ipv6_addr = *(struct ipv6_addr __far*)addr->sockaddr_in6.sin6_addr;
    // dirty hack, must be a near ptr and returns random values on listening
    // sockets...
    if(address_in_same_ipv6_subnet(&ipv6_addr))
      ptr = &ipv6_local_addr;
    else
      ptr = &ipv6_global_addr;
    _fmemcpy(addr->sockaddr_in6.sin6_addr, ptr, 16);
  }

  return 0;
}

#define SOCKET_FLAG_PEEK 0x01

int socket_read(int socket_desc, char __far *buff, unsigned count)
{
  int socket = socket_descriptor_to_socket(socket_desc);
  unsigned __far *write_offset = &sockets[socket]->in_buff_write_offset;
  unsigned __far *read_offset = &sockets[socket]->in_buff_read_offset;
  unsigned data_count;

  if(socket == -1)
  {
    if((socket = socket_descriptor_to_open_port(socket_desc)) == -1)
    {
      error_code = ENOTCONN;
      return -1;
    }
    else
    {
      if(open_ports[socket]->socket_queue_count && count >= 2)
      {
        *(unsigned short __far*)buff = pop_socket_queue(socket);
        return 2;
      }
      return 0;
    }
  }

  if(!sockets[socket]->status || (!sockets[socket]->data_pending && sockets[socket]->status == 3))
  {
    error_code = ENOTCONN;
    return -1;
  }

  data_count = *write_offset - *read_offset;

  if(!data_count)
  {
    service_packets(); // try to get more data
    if(data_count = (*write_offset - *read_offset)) goto return_byte;
    if(sockets[socket]->data_pending) delete_socket(socket, 1);
    return 0;
  }

  return_byte:

  if(sockets[socket]->raw)
  {
    ringcopy((unsigned char __far*)&data_count, sockets[socket]->in_buff, *read_offset, SOCKET_IN_BUFF_SIZE,
      sizeof(unsigned));
    if(data_count > count) return -1;
    *read_offset += sizeof(unsigned);
  }

  if(data_count < count) count = data_count;

  ringcopy(buff, sockets[socket]->in_buff, *read_offset, SOCKET_IN_BUFF_SIZE,
           count);

  if(!(sockets[socket]->flags & SOCKET_FLAG_PEEK))
    *read_offset += count;

  if(sockets[socket]->recv.zerowindow)
    service_packets();

  return count;
}

int socket_write(int socket_desc, char __far *buff, unsigned count)
{
  int socket = socket_descriptor_to_socket(socket_desc);
  unsigned __far *write_offset = &sockets[socket]->out_buff_write_offset;
  unsigned __far *read_offset = &sockets[socket]->out_buff_read_offset;
  unsigned space_left;
  signed space_after_op;

  if(socket == -1 || !sockets[socket]->status || sockets[socket]->status == 3 || sockets[socket]->send_pending)
  {
    error_code = ENOTCONN;
    return -1;
  }

  if(sockets[socket]->raw)
  {
    return raw_packetizer(socket, buff, count);
  }

  space_left = SOCKET_OUT_BUFF_SIZE - (*write_offset - *read_offset);

  if(space_left < OUT_BUFF_MIN_FREE_SPACE)
  {
    service_packets(); // try to make more space
    if(!(space_left = SOCKET_OUT_BUFF_SIZE - (*write_offset - *read_offset)))
    {
      error_code = ENOBUFS;
      return 0;
    }
  }

  if(!sockets[socket]->status)
  {
    error_code = ENOTCONN;
    return -1;
  }

  space_after_op = space_left - count;

  if(space_after_op < 0)
  {
    count = space_left;
    space_after_op = 0;
  }

  if(space_left > OUT_BUFF_MIN_FREE_SPACE && space_after_op <= OUT_BUFF_MIN_FREE_SPACE)
  {
    count -= OUT_BUFF_MIN_FREE_SPACE - space_after_op;
  }

  copy2ring(sockets[socket]->out_buff, buff, *write_offset,
           SOCKET_OUT_BUFF_SIZE, count);
  *write_offset += count;

  if(space_after_op <= OUT_BUFF_MIN_FREE_SPACE)
    service_packets();

  return count;
}

int socket_modflags(int, unsigned char, unsigned char);
#pragma aux socket_modflags value [ax] parm [bx] [ch] [cl];

int socket_modflags(int socket_desc, unsigned char func, unsigned char flags)
{
  int socket = socket_descriptor_to_socket(socket_desc);

  if(socket == -1)
  {
    error_code = ENOTCONN;
    return -1;
  }

  if(func & 0x80)
    sockets[socket]->flags = flags;

  return sockets[socket]->flags;
}

static void tcp4_verify_send(unsigned socket, struct tcp4_header *header)
{
  unsigned p;
  unsigned long all_ack = sockets[socket]->send.iss
                      + sockets[socket]->send.count
                      + sockets[socket]->send.ack_waiting;
  signed long i = all_ack - header->ack_number;
  unsigned latency;
  unsigned char ploss_and;

  if(i)
  {
    if((p = socket_packet_in_buffers(socket, header->ack_number)) != -1)
      mark_outpacket_sent(p);
    else if(i < 0)
      tcp4_reset_socket(socket);
    return;
  }

  socket_delete_buffers(socket);

  latency = update_latency(socket, sockets[socket]->send.last_send) >> 16;
  // lower latency == faster acceleration
  ploss_and = 0x1F >> latency;

  if(sockets[socket]->ploss && !(--sockets[socket]->ploss_chg & ploss_and))
    sockets[socket]->ploss--;

  sockets[socket]->out_buff_read_offset += sockets[socket]->send.ack_waiting;
  sockets[socket]->send.zerowindow = !(sockets[socket]->send.window = header->window);
  sockets[socket]->send.count += sockets[socket]->send.ack_waiting;
  sockets[socket]->status = 1;
  sockets[socket]->time = ttime(0);    
  sockets[socket]->send.retries = 0;
}

static int push_socket_queue(unsigned open_port, unsigned sock_desc)
{
  unsigned __far *count = &open_ports[open_port]->socket_queue_count;
  if(*count < SOCKET_QUEUE_LEN)
  {
    open_ports[open_port]->socket_queue[*count] = sock_desc;
    ++*count;
    return 0;
  }
  return -1;
}

static int pop_socket_queue(unsigned open_port)
{
  unsigned __far *count = &open_ports[open_port]->socket_queue_count;
  int retval;

  retval = open_ports[open_port]->socket_queue[0];
  --*count;
  _fmemcpy(&open_ports[open_port]->socket_queue[0],
            &open_ports[open_port]->socket_queue[1],
            *count *sizeof(unsigned));
  return retval;
}

// return 1 if the socket required wakeup from power-saving, else return 0
static int tcp4_packetizer(unsigned socket)
{
  unsigned __far *write_offset = &sockets[socket]->out_buff_write_offset;
  register unsigned read_offset;
  struct tcp4_data data;
  unsigned char buff[SOCKET_OUT_BUFF_SIZE];
  unsigned payload_size, data_count;
  unsigned long time = ttime(0);
  unsigned retval = 0;

  if(!sockets[socket]->status)
  {
    if(time > (sockets[socket]->time + TCP_RETRANSMISSION_TIME))
    {
      if(sockets[socket]->send.retries < MAX_RETRIES)
      {
        if(sockets[socket]->accepted)
          tcp4_resynack_socket(socket);
        else
          resend_syn4(socket);
      }
      else if(time > (sockets[socket]->time + TCP_TIMEOUT))
        delete_socket(socket, 1);     
      return 1;
    }
    return 0;
  }

  else if(sockets[socket]->status == 3)
  {
    if(time > (sockets[socket]->time + TCP_TIMEOUT))
    {
      delete_socket(socket, 1);
      return 1;
    }
    return 0;
  }

#ifdef TCP_RETRANS_ANSWER_DELAY
  if(sockets[socket]->first_ack_retrans
     && time > (sockets[socket]->time + TCP_RETRANS_ANSWER_DELAY))
  {
    sockets[socket]->first_ack_retrans = 0;
    tcp4_reack_socket(socket);
    return 1;
  }
#endif

  if(sockets[socket]->user_closed)
    if(time > (sockets[socket]->user_closed + CLOSE_TIMEOUT))
    {
      tcp4_reset_socket(socket);
      return 0;
    }

/*  if(time > (sockets[socket]->time + TCP_RETRANSMISSION_TIME))
  {
    if(resend_outpacket4(socket) == 1)
      return 1;
  }*/
  retval = resend_outpacket4(socket);

  data.data = buff;
  do
  {
    data.only_save = 0;
    read_offset = sockets[socket]->out_buff_read_offset;

    if(sockets[socket]->status == 2)
    {
      payload_size = sockets[socket]->send.ack_waiting;
      if((signed long)(clock_tick_count() - sockets[socket]->send.last_send) > (sockets[socket]->send.latency|1))
      {
        // resend
        if(sockets[socket]->send.retries < MAX_RETRIES)
          sockets[socket]->ploss++;
        else if(time > (sockets[socket]->time + TCP_TIMEOUT))
        {
          delete_socket(socket, 1);
          return 0;
        }
        else return 0;
      }
      else if(sockets[socket]->send.window && out_buffers_free(socket))
        data.only_save = 1;
      else return retval;
    }
    else  
    {
      if(!(data_count = *write_offset - read_offset))
      {
        // no data to be sent
        if(sockets[socket]->send_pending)
          tcp4_close_socket(socket);
        else if(sockets[socket]->recv.zerowindow && get_window_size(socket))
          tcp4_send_ack(socket);
        if(sockets[socket]->time < time-1)
          sockets[socket]->ploss = INIT_PLOSS;
        return retval;
      }
      if(data_count < sockets[socket]->last_frame_size)
      {
        // avoid sending small frames
        sockets[socket]->last_frame_size >>= 2;
        return retval;
      }

      if(!(payload_size = sockets[socket]->send.window))
      {
        if(sockets[socket]->send.zerowindow)
        {
          payload_size = sockets[socket]->send.window = 1;
          goto try_send_one_byte;
        }
        else
          return retval;
      }
  
      if(data_count < payload_size) payload_size = data_count;

      try_send_one_byte:
      sockets[socket]->send.window -= payload_size;
      sockets[socket]->last_frame_size = sockets[socket]->send.ack_waiting = payload_size;
    }

    ringcopy(data.data, sockets[socket]->out_buff, read_offset, SOCKET_OUT_BUFF_SIZE, payload_size);
    data.len = payload_size;
 
    if(sockets[socket]->status == 2 && !sockets[socket]->send.zerowindow)
      sockets[socket]->send.retries++;

    retval++;

    if(tcp4_push_socket(socket, &data))
      break;
  }
  while(sockets[socket]->status == 1);

  return retval;
}

static int udp4_packetizer(unsigned socket)
{
  unsigned write_offset = sockets[socket]->out_buff_write_offset;
  unsigned read_offset = sockets[socket]->out_buff_read_offset;
  struct udp4_header header;
  unsigned char buff[SOCKET_OUT_BUFF_SIZE]; 
  struct udp_data data;
  unsigned payload_size;

  if(!(payload_size = write_offset - read_offset))
    return 0; // no data to be sent

  data.data = buff;

  header.src_port = sockets[socket]->local_port;
  header.dst_port = sockets[socket]->remote_port;

  ringcopy(data.data, sockets[socket]->out_buff, read_offset, SOCKET_OUT_BUFF_SIZE, payload_size);
  data.len = payload_size;
  sockets[socket]->out_buff_read_offset = write_offset;

  if(!sockets[socket]->ipv6)
    send_udp4(sockets[socket]->ip_addr.ipv4_addr, &header, &data);
  else
    send_udp6(sockets[socket]->ip_addr.ipv6_addr, &header, &data);

  return 1;
}

// return count of packets that needed wakeup from power-saving mode
unsigned service_packets(void)
{
  int n;
  unsigned retval = 0;
  unsigned long time;

  while(loopback.has_data)
  {
    parse_loopback();
    retval++;
  }

  // read incoming data into buffers
  while((n = packet_ready()) != -1)
  {
    retval++;
    parse_incoming_packet(n);
  }

  n = MAX_SOCKETS;

  while(n--)
  {
    if(transmit_not_ready()) break;
    // send outgoing data
    if(sockets[n])
    {
      if(sockets[n]->udp)
        retval += udp4_packetizer(n);
      else
        retval += tcp4_packetizer(n);
    }
  }


  time = ttime(0);
  if((unsigned)lease.status <= 2)
  {
    if(!lease.status && time > lease.lease_end_time)
    {
      lease.status = 2;
      lease.request_time = time;
      lease.time_elapsed = 0;
      lease.lease_end_time = -1;
    }
    if(lease.status && time > lease.request_time + lease.time_elapsed)
      send_dhcp_request();
  }

  if((unsigned)lease6.status < 2)
  {
    if(!lease6.status && time > lease6.lease_end_time)
    {
      lease6.status = 2;
      lease6.request_time = time;
      lease6.time_elapsed = 0;
      lease6.lease_end_time = -1;
      lease6.transaction_id = random32() & 0xFFFFFF00;
    }
    else if(lease6.status && time > lease6.request_time + lease6.time_elapsed)
      send_dhcp6_request();
  }

  if(lease6.status == 2 && time > lease6.request_time + lease6.time_elapsed)
    if(lease6.use_dhcp6)
      renew_dhcp6();

  return retval;
}



int close_socket(int socket_desc)
{
  int socket = socket_descriptor_to_socket(socket_desc);
  if(socket == -1)
    if((socket = socket_descriptor_to_open_port(socket_desc)) == -1)
    {
      error_code = ENOTCONN;
      return -1;
    }
    else return close_port(socket);

  if(sockets[socket]->raw)
    goto delete;  
  if(!sockets[socket]->udp)
  {
    sockets[socket]->user_closed = ttime(0);
    tcp4_close_socket(socket);
  }
  else
  {
    udp4_packetizer(socket);
    delete:delete_socket(socket, 1);
  }

  return 0;
}
