#define tcp4_pseudoheader(s, d, l) ipv4_pseudoheader(s, d, l, 6)
#define tcp6_pseudoheader(s, d, l) ipv6_pseudoheader(s, d, l, 6)

static void parse_tcp4(unsigned char __far *packet, struct tcp4_header *header,
                       struct tcp4_data *tcp4_data, unsigned packet_len, 
                       union ip_addr *src_addr, unsigned char *sender_mac, char ipv6)
{
  unsigned short __far *shortptr = (unsigned short __far*)packet;
  unsigned long __far *longptr = (unsigned long __far*)packet;
  int n;
  int o;
  int s;

  header->src_port = switch_endianness_s(shortptr[0]);
  header->dst_port = switch_endianness_s(shortptr[1]);
  header->seq_number = switch_endianness_l(longptr[1]);
  header->ack_number = switch_endianness_l(longptr[2]);
  header->data_offset = packet[12] >> 2;
  if(packet[13] & 0x01) header->flags.fin = 1;
  if(packet[13] & 0x02) header->flags.syn = 1;
  if(packet[13] & 0x04) header->flags.reset = 1;
  if(packet[13] & 0x08) header->flags.push = 1;
  if(packet[13] & 0x10) header->flags.ack = 1;
  if(packet[13] & 0x20) header->flags.urg = 1;
  header->window = switch_endianness_s(shortptr[7]);
  header->urg_ptr = switch_endianness_s(shortptr[9]);

  if(!ipv6 ? 
      (src_addr->ipv4_addr.p1 != 127 &&
        udp4_checksum(packet, tcp4_pseudoheader(src_addr->ipv4_addr, ipv4_my_addr, packet_len), packet_len, 6))
    : (!is_localhost6(&src_addr->ipv6_addr) && 
        udp4_checksum(packet, tcp6_pseudoheader(src_addr->ipv6_addr, *ipv6_my_addr, packet_len), packet_len, 20)))
    {
      rx_error_count++;
      return;
    }

  n = !ipv6
      ?  find_socket_by_ports_and_remote_addr4(header->dst_port, header->src_port, src_addr->ipv4_addr, 0)
      :  find_socket_by_ports_and_remote_addr6(header->dst_port, header->src_port, src_addr->ipv6_addr, 0);
  if(n == -1)
  {
    o = find_open_port_by_port(header->dst_port, 0);
    if(o == -1) return;
    if(open_ports[o]->ipv_matters && open_ports[o]->ipv6 != ipv6) return;

    if(header->flags.syn)
    {
      // make new socket      
      s = find_free_socket_index();
      if(s == -1)
        return; // out of resources
      sockets[s]->accepted = 1;
      _fmemcpy(&sockets[s]->ip_addr, src_addr, sizeof(union ip_addr));
      sockets[s]->send.iss = random32();
      sockets[s]->remote_port = header->src_port;
      sockets[s]->local_port = header->dst_port;
      sockets[s]->time = ttime(0);
      sockets[s]->ipv6 = ipv6;
      // save remote address
      _fmemcpy(sockets[s]->remote_addr, sender_mac, 6);      
      // send SYN, ACK
      tcp4_synack_socket(s, header);
      return;
    }
    else
    {
      // socket not found
      return;
    }
  }
  if(sender_mac_changed(n, sender_mac, &src_addr))    
    return;  // sender's physical address changed, drop packet

  if(header->flags.reset)
  {
    delete_socket(n, 1);
//    error_code = ECONNRESET;        
  }
  else if(header->flags.ack)
  {
    sockets[n]->send.retries = 0;
    switch(sockets[n]->status)
    {
      case 0:
        sockets[n]->send.window = header->window;
        update_latency(n, sockets[n]->send.last_send);
        if(header->flags.syn)
        {
          if(sockets[n]->accepted)
            tcp4_resynack_socket(n);
          else
          {
            sender_mac_changed(n, sender_mac, &src_addr);
            tcp4_ack_socket(n, header);
          }
          return;
        }
        else if(sockets[n]->accepted)
        {
          if(!push_socket_queue(find_open_port_by_port(header->dst_port, 0), sockets[n]->socket_descriptor))
            goto recv;
        }
        break;
      case 2:
        if(header->flags.syn)
        {
          if(header->seq_number == sockets[n]->recv.irs)
#ifndef TCP_RETRANS_ANSWER_DELAY
            tcp4_reack_socket(n);
#else
            sockets[n]->time = ttime(0);
            sockets[n]->first_ack_retrans = 1;
#endif
          return;
        }
      case 1:
        recv:tcp4_recv_socket(n, header, &packet[header->data_offset], packet_len - header->data_offset);
      case 3:
        if(header->flags.fin)
          tcp4_remote_close(n, header);
        break;
    }
  }
}

static unsigned build_tcp4_header(unsigned char __far *packet, struct tcp4_header *header)
{
  unsigned short __far *shortptr = (unsigned short __far*)packet;
  unsigned long __far *longptr = (unsigned long __far*)packet;
  unsigned header_len = 20;

  shortptr[0] = switch_endianness_s(header->src_port);
  shortptr[1] = switch_endianness_s(header->dst_port);
  longptr[1] = switch_endianness_l(header->seq_number);
  longptr[2] = switch_endianness_l(header->ack_number);
  packet[12] = header_len << 2;
  packet[13] = 0;
  if(header->flags.fin) packet[13] |= 0x01;
  if(header->flags.syn) packet[13] |= 0x02;
  if(header->flags.reset) packet[13] |= 0x04;
  if(header->flags.push) packet[13] |= 0x08;
  if(header->flags.ack) packet[13] |= 0x10;
  if(header->flags.urg) packet[13] |= 0x20;
  shortptr[7] = switch_endianness_s(header->window);
  shortptr[8] = 0; // checksum
  shortptr[9] = switch_endianness_s(header->urg_ptr);


  return header_len;  
}

static unsigned build_tcp4_packet(unsigned char __far *packet, struct tcp4_header *header, struct tcp4_data *data, union ip_addr *dst_addr, char ipv6)
{
  unsigned short __far *checksum = (unsigned short __far*)&packet[16];
  unsigned header_len;
  unsigned short len = 0;

  header_len = build_tcp4_header(packet, header);
  if(data)
  {
    _fmemcpy(&packet[header_len], data->data, data->len);
    len = data->len;
  }

  if(!ipv6)    
    *checksum = udp4_checksum(packet, tcp4_pseudoheader(ipv4_my_addr, dst_addr->ipv4_addr, header_len + len), header_len + len, 6);
  else
    *checksum = udp4_checksum(packet, tcp6_pseudoheader(*ipv6_my_addr, dst_addr->ipv6_addr, header_len + len), header_len + len, 20);

  return header_len + len;
}

static int send_tcp4(unsigned socket, struct tcp4_header *header, struct tcp4_data *data, char __far *dest_mac)
{
  unsigned total_length;
  unsigned ipv4_header_length;
  struct ipv4_header ipv4_header;  
  unsigned char __far *packet;
  unsigned char __far *payload;
  struct ipv4_addr dst = sockets[socket]->ip_addr.ipv4_addr;

  packet = get_free_async_buff();
  payload = &packet[14];

  init_ipv4_packet_header(dst, packet, dest_mac);

  init_ipv4_header_struct(&ipv4_header, 6, dst);

  ipv4_header_length = build_ipv4_header(payload, &ipv4_header);

  total_length = ipv4_header_length + build_tcp4_packet(&payload[ipv4_header_length], header, data, (union ip_addr*)&dst, 0);

  ipv4_inject_lengthinfo(payload, total_length);
  ipv4_inject_header_checksum(payload, ipv4_header_length);

  if(data && payload[16] != 127)
  {
    if(sockets[socket]->send.window)
    {
      add_outpacket(socket, packet, total_length+14, data->len);
    }
    if(data->only_save)
    {
      outpktbuff_inuse = 0xFF;
      return 0;
    }
  }
  sockets[socket]->send.last_send = clock_tick_count();

  return send_ip_packet(packet, total_length+14);
}

static int send_tcp(unsigned socket, struct tcp4_header *header, struct tcp4_data *data, char __far *dest_mac)
{
  if(!sockets[socket]->ipv6)
    return send_tcp4(socket, header, data, dest_mac);
  else
    return send_tcp6(socket, header, data, dest_mac);
}
